package jacorb.imr;

import jacorb.imr.RegistrationPackage.*;
import jacorb.imr.AdminPackage.*;

/**
 * This class stores information about a POA. It also provides methods 
 * for reactivation, conversion, and for waiting for reactivation.
 *
 * @author Nicolas Noffke
 * 
 * $Log: ImRPOAInfo.java,v $
 * Revision 1.7  1999/11/25 16:04:54  brose
 * cosmetics
 *
 * Revision 1.6  1999/11/09 10:59:26  brose
 * new GUI version
 *
 * Revision 1.4  1999/10/30 16:02:57  noffke
 * Fixed bug: servers could be restarted multiple times when serveral
 * clients tried to restart them simultaneously.
 *
 * Revision 1.3  1999/10/29 19:24:29  noffke
 * Cosmetics, now fully commented , adapted to new/altered methods from the
 * interface
 *
 *
 */

public class ImRPOAInfo implements java.io.Serializable{
    protected int m_port;
    protected ImRServerInfo m_server;
    protected String m_host;
    protected String m_name;
    protected boolean m_active;
    protected static long m_timeout = 120000; // 2 min.

    static{
	// read in timeout from Environment
	String _tmp = jacorb.util.Environment.getProperty("jacorb.imr.timeout");
	if (_tmp != null)
	    try{
		m_timeout = Integer.parseInt(_tmp);
	    }catch(Exception _e){}
    }
    
    /**
     * The constructor of this class.
     *
     * @param name the POAs name.
     * @param host the POAs host.
     * @param port the port the POA listens on.
     * @param server the server the POA is associated with.
     * @exception IllegalPOAName thrown when <code>name</code> is 
     * <code>null</code> or of length zero.
     **/
    public ImRPOAInfo(String name, String host, int port, ImRServerInfo server) 
	throws IllegalPOAName {

	//super(name, host);

	if (name == null || name.length() == 0)
	    throw new IllegalPOAName(name);

	m_name = name;
	m_host = host;
	m_port = port;
	m_server = server;
	m_active = true;
    }

    /**
     * "Converts" this Object to an instance of the POAInfo class.
     * 
     * @return a POAInfo object.
     **/    
    public POAInfo toPOAInfo(){
	return new POAInfo(m_name, m_host, m_port, m_server.m_name, m_active); 
    }

    /**
     * Reactivates this POA, i.e. sets it to active and unblocks any waiting threads.
     *
     * @param host the POAs new host.
     * @param port the POAs new port.
     **/
    public synchronized void reactivate(String host, int port){
	m_host = host;
	m_port = port;
	m_active = true;	
	m_server.m_active = true;
	m_server.m_restarting = false;
	notifyAll();
    }

    /**
     * This method blocks until the POA is reactivated, or the
     * timeout is exceeded.
     * 
     * @return false, if the timeout has been exceeded, true otherwise.
     **/        
    public synchronized boolean awaitActivation(){
	while(!m_active){
	    try{
		long _sleep_begin = System.currentTimeMillis();
		wait(m_timeout);
		if (!m_active && (System.currentTimeMillis() - _sleep_begin) > m_timeout)
		{
		    jacorb.util.Debug.output(2, "awaitActivation, time_out");
		    return false;
		}
	    }catch (java.lang.Exception _e){
		jacorb.util.Debug.output(2, _e);
	    }
	}
        jacorb.util.Debug.output(2, "awaitActivation, returns true");

	return true;
    }
} // ImRPOAInfo


